/**
* \file: automounter_types.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AUTOMOUNTER_TYPES_H_
#define AUTOMOUNTER_TYPES_H_

#ifdef __cplusplus
extern "C" {
#endif

//this header file has been adapted to fully support cpp programs linking against automounter.so.
//This define indicates that this version of automounter / platform is compatible and can be used.
//This flag is evaluated by MCs device manager components to decide if it can use the binary interface
//or it should stick with the old interface depending on the platform it is build on top of.
#define CPP_COMPATIBLE_VERSION

#include <stdbool.h>
#include <stddef.h>

/**
 * \brief defines possible states of a mass storage device
 * See device state machine in the design document for further details.
 */
typedef enum __attribute__((packed)) device_state_t
{
	/**
	 * \brief A device has been detected.
	 * The automounter starts working on the device.
	 */
	DEVICE_DETECTED			=0,

	/**
	 * \brief The automounter is automatically mounting partitions found on the device.
	 * This state is either entered after a device has been connected or a medium has been inserted.
	 */
	DEVICE_AUTOMOUNTING		=1,

	/**
	 * \brief The automounter finished working on all partitions of the device
	 */
	DEVICE_AUTOMOUNTED		=2,

	/**
	 * \brief The automounter starts unmounting all partitions of the device triggered by a request
	 */
	DEVICE_UNMOUNTING		=3,

	/**
	 * \brief The automounter finished unmounting all partitions of the device.
	 */
	DEVICE_UNMOUNTED		=4,

	/**
	 * \brief The device has been removed.
	 */
	DEVICE_INVALID			=5,

	/**
	 * \brief There is no media in the device. This state is either entered after a new device has been detected without
	 * any medium inside or the medium has been removed.
	 */
	DEVICE_NOMEDIA			=6,

	/**
	 * \brief A transient state entered while the model of the device is created.
	 */
	_UNKNOWN_DEVICE_STATE	=7,

	//just to ensure that always the same number of bytes is used for this enum even if it is extended
	__SIZE_SCALER_DS		=0xFF //do not use
} device_state_t;

/**
 * \brief defines possible states of a partition of a mass storage device
 * See partition state machine in the design document for further details.
 */
typedef enum __attribute__((packed)) partition_state_t
{
	/**
	 * \brief A new partition is detected.
	 * Partitions appear either on a newly connected device or on an inserted medium.
	 */
	PARTITION_DETECTED			=0,

	/**
	 * \brief The automounter starts automatically mounting the partition.
	 */
	PARTITION_MOUNTING			=1,

	/**
	 * \brief The automounter finished mounting the partition.
	 * This state is either entered when a newly detected partition
	 * has been automatically mounted or as a result of a remount request.
	 */
	PARTITION_MOUNTED			=2,

	/**
	 * \brief The automounter starts unmounting the partition triggered by a request
	 */
	PARTITION_UNMOUNTING		=3,

	/**
	 * \brief The partition is not mounted.
	 */
	PARTITION_UNMOUNTED			=4,

	/**
	 * \brief The automounter is currently remounting the partition triggered by a request
	 */
	PARTITION_REMOUNTING		=5,

	/**
	 * \brief The medium or the device to which the partition belongs has been removed
	 * The partition and especially the mount point is not valid any longer.
	 */
	PARTITION_INVALID			=6,

	/**
	 * \brief The detected partition is not supported by the automounter
	 * Possible reasons are defined in the \ref partition_unsupported_reason_t enumeration. The field unsuppored_reason
	 * in the \ref partition_info_t data structure is set respectively.
	 */
	PARTITION_UNSUPPORTED		=7,

	/**
	 * \brief The automounter failed to mount the partition.
	 * The error code can be found in the error field of the \ref partition_info_t data structure.
	 */
	PARTITION_MOUNT_ERR			=8,

	/**
	 * \brief A transient state entered while the model of the partition is created.
	 */
	_UNKNOWN_PARTITION_STATE	=9,

	//just to ensure that always the same number of bytes is used for this enum even if it is extended
	__SIZE_SCALER_PS			=0xFF //do not use
} partition_state_t;

/**
 * \brief Defines an enumeration of reasons that lead to a partition entering the state PARTITION_UNSUPPORTED.
 * The \ref partition_info_t data structure contains a field with one of the reasons set.
 */
typedef enum __attribute__((packed)) partition_unsupported_reason_t
{
	/**
	 * \brief The partition is supported.
	 * Hence, it is not in state PARTITION_UNSUPPORTED when this value is set.
	 */
	_SUPPORTED					=0,

	/**
	 * \brief The file system of the partition is either unknown or not in the list of supported file systems in the
	 * automounter's configuration file.
	 */
	UNSUPPORTED_FILESYSTEM		=1,

	/**
	 * \brief No file system is found on a partition.
	 */
	NO_FILESYSTEM				=2,

	/**
	 * \brief The medium inside the mass storage device is an audio cd (cdda)
	 */
	AUDIO_CD					=3,

	/**
	 * \brief The medium inside the mass storage device is a mixed mode audio cd and the automounter is configured to not
	 * mount the data part of the cd (see automounter.conf).
	 */
	AUDIO_MIXED_MODE			=4,

	/**
	 * \brief The partition is blacklisted in the automounter's configuration file.
	 */
	BLACKLISTED					=5,

	//just to ensure that always the same number of bytes is used for this enum even if it is extended
	__SIZE_SCALER_PUR			=0xFF //do not use
} partition_unsupported_reason_t;

/**
 * \brief defines error codes that are returned by API functions in case of failures.
 * A rough descriptions of the individual error codes is already given here.
 * Concrete meaning of the code are described with the individual API functions.
 */
typedef enum __attribute__((packed)) error_code_t
{
	/**
	 * \brief no failures
	 */
	RESULT_OK						=0,

	/**
	 * \brief An API function has been used wrongly.
	 * e.g.: API functions are called while the shared library is in an
	 * incompatible state
	 */
	RESULT_INVALID					=1,

	/**
	 * \brief Return in case of any kind of resource issues (such as memory problems).
	 */
	RESULT_NORESOURCE				=2,

	/**
	 * \brief The automounter was unable to mount a partition.
	 */
	RESULT_MOUNT_ERR				=3,

	/**
	 * \brief The automounter was unable to create a mount point for a partition
	 */
	RESULT_INVALID_MOUNTPOINT		=4,

	/**
	 * \brief The automounter was unable to unount a partition (internally used)
	 */
	RESULT_UMOUNT_ERR				=5,

	/**
	 * \brief The partition requested to be unmounted is not mounted.
	 */
	RESULT_NOT_MOUNTED				=6,

	/**
	 * \brief internally used only
	 */
	RESULT_INVALID_ARGS				=7,

	/**
	 * \brief internally used only
	 */
	RESULT_HELP_PRINTED				=8,

	/**
	 * \brief internally used only
	 */
	RESULT_SOCKET_ERR				=9,

	/**
	 * \brief internally used only
	 */
	RESULT_MSG_PARTLY_PROCESSED		=10,

	/**
	 * \brief internally used only
	 */
	RESULT_SOCKET_MSG_CORRUPT		=11,

	/**
	 * \brief internally used only
	 */
	RESULT_SOCKET_CLOSED			=12,

	/**
	 * \brief internally used only
	 */
	RESULT_CONNECT_TO_DAEMON_ERR	=13,

	/**
	 * \brief  A connection is requested but the daemon is not running.
	 */
	RESULT_DAEMON_NOT_RUNNING		=14,

	/**
	 * \brief The application has not the access rights to access the automounter control socket.
	 * No connection is possible.
	 */
	RESULT_ACCESS_RIGHTS			=15,

	/**
	 * \brief internally used only
	 */
	RESULT_READ_DEV_ERR				=16,

	//just to ensure that always the same number of bytes is used for this enum even if it is extended
	__SIZE_SCALER_EC				=0xFF //do not use
} error_code_t;

/**
 * \brief Represents a device in the shared library.
 *
 * This data structure contains information about a device. It is passed to all callback functions
 * that are related to a device.
 */
typedef struct __attribute__((aligned(8),packed)) device_info_t
{
	/**
	 * \brief An id used to identify the device in the pool of devices currently managed by the automounter.
	 *
	 * This id is required to request an unmount of the device. Please note that this id does not identify
	 * the device itself. One device connected different times can lead to different values of this attribute.
	 */
	const char *interface_id;

	/**
	 * \brief the state of the device
	 */
	device_state_t state;

	/**
	 * \brief A heuristically generated string which can be used to identify a device
	 *
	 * This attribute can be used to identify the device itself. It is guaranteed that for one and the same device
	 * always the same identifier is created. But it can not be guaranteed that there exist no two devices that lead
	 * to the same id.
	 */
	const char *identifier;

	/**
	 * \brief The number of partitions already detected for the device.
	 *
	 * If the device is in state DEVICE_AUTOMOUNTED,
	 * this number represents all partitions found on the device / medium in a device.
	 */
	int detected_partition_cnt;

	/**
	 * \brief A string that describes the device type.
	 *
	 * The generic mass storage handler differentiates between "disk" and "cd" here. Other device might define
	 * additional values here.
	 */
	const char *device_type;

	/**
	 * \brief A string that can be used to identify the device handler which handles the device.
	 *
	 * The generic mass storage handler set this string to "GENERAL_MSD". Other mass storage handler need to define own
	 * identifiers.
	 */
	const char *device_handler_id;

	/**
	 * \brief Contains the size of the instance of the data structure
	 *
	 * This attribute is meant for upwards and downwards compatibility. Depending on the version of the shared library
	 * you are linked against, this size might be larger or smaller then the size of the data structure defined in the
	 * header file you have used to build your application. The minimal set of attributes which is guaranteed to be
	 * delivered with any version ends with the attribute extension_bitmask. Newer versions of the library might extend
	 * the data structure by additional attributes. Their presence can be determined by evaluating the extension_bitmask.
	 * Each available extended attribute sets a respective bit flag in the mask.
	 */
	size_t data_structure_real_size;

	/**
	 * \brief A bit mask defining the extended attributes contained in this instance of the data structure.
	 *
	 * This attribute is meant for upwards and downwards compatibility. Depending on the version of the shared library
	 * you are linked against, a set of extended attributes might be contained in this instance of the data structure.
	 * The minimal set of attributes which is guaranteed to be delivered with any version ends with the attribute
	 * extension_bitmask. Newer versions of the library might extend the data structure by additional attributes.
	 * Their presence can be determined by evaluating this bit mask. Each available extended attribute sets a
	 * respective bit flag in the mask.
	 */
	long extension_bitmask;

} device_info_t;


/**
 * \brief Represents a partition in the shared library.
 *
 * This data structure contains information about a partition. It is passed to all callback functions
 * that are related to a partition.
 */
typedef struct __attribute__((aligned(8),packed)) partition_info_t
{
	/**
	 * \brief An id used to identify the partition in the pool of partitions currently managed by the automounter.
	 *
	 * This id is required to request an unmount or remount of the partition. Please note that this id does not identify
	 * the partition itself. One device connected different times can lead to different values of this attribute.
	 */
	const char *interface_id;

	/**
	 * \brief the state of the partition
	 */
	partition_state_t state;

	/**
	 * \brief A reason stating why the partition is unsupported
	 * It is set when the partition enters the state PARTITION_UNSUPPORTED.
	 */
	partition_unsupported_reason_t unsupported_reason;

	/**
	 * \brief An error code stating why the partition can not be mounted.
	 * It is set when the partition enters the state mount error.
	 */
	error_code_t error;

	/**
	 * \brief A flag stating whether or not a partition is mounted writable.
	 */
	bool mounted_writable;

	/**
	 * \brief The partition number of the partition.
	 *
	 * The value here depends on the type of the mass storage device connected. In case the msd contains a
	 * partition table, this number is set to the number of the partition in the table. Devices without a partition
	 * table are represented by only one partition so this number is set always to 1. Other device handlers might set
	 * this value to their own interpretation of the numbering of different partitions.
	 */
	int partition_no;

	/**
	 * \brief A heuristically generated string which can be used to identify a partition
	 *
	 * This attribute can be used to identify the partition itself. It is guaranteed that for one and the same
	 * partition (as long as the content has not been changed) always the same identifier is created.
	 * But it can not be guaranteed that there exist no two different partitions that lead to the same id.
	 */
	const char *identifier;

	/**
	 * \brief Contains the mount source, which is finally passed to the mount system call to mount the partition.
	 *
	 * In case of a generic Linux block device, this is always a path to the device node that is exported by the kernel
	 * driver to access the partition. Other device handlers can own mount sources depending on how the mass storage
	 * device is finally mounted.
	 */
	const char *mount_src;

	/**
	 * \brief Contains the identifier of the file system found on the partition.
	 */
	const char *mount_fs;

	/**
	 * \brief Contains the mount point. to which the partition is mounted.
	 *
	 * This value is only valid, when the partition is in state PARTITION_MOUNTED
	 */
	const char *mount_point;

	/**
	 * \brief contains the size of the instance of the data structure
	 *
	 * This attribute is meant for upwards and downwards compatibility. Depending on the version of the shared library
	 * you are linked against, this size might be larger or smaller then the size of the data structure defined in the
	 * header file you have used to build your application. The minimal set of attributes which is guaranteed to be
	 * delivered with any version ends with the attribute extension_bitmask. Newer versions of the library might extend
	 * the data structure by additional attributes. Their presence can be determined by evaluating the extension_bitmask.
	 * Each available extended attribute sets a respective bit flag in the mask.
	 */
	size_t data_structure_real_size;

	/**
	 * \brief A bit mask defining the extended attributes contained in this instance of the data structure.
	 *
	 * This attribute is meant for upwards and downwards compatibility. Depending on the version of the shared library
	 * you are linked against, a set of extended attributes might be contained in this instance of the data structure.
	 * The minimal set of attributes which is guaranteed to be delivered with any version ends with the attribute
	 * extension_bitmask. Newer versions of the library might extend the data structure by additional attributes.
	 * Their presence can be determined by evaluating this bit mask. Each available extended attribute sets a
	 * respective bit flag in the mask.
	 */
	long extension_bitmask;

} partition_info_t;

/**
 * \brief defines log levels for the shared library and the automounter daemon
 */
typedef enum __attribute__((packed)) logger_loglevel_t
{
	/**
	 * \brief Nothing is logged at all
	 */
	LOGGER_LEVEL_NONE		=0,

	/**
	 * \brief Only errors are logged
	 */
	LOGGER_LEVEL_ERROR		=1,

	/**
	 * \brief Error and info message are logged.
	 */
	LOGGER_LEVEL_INFO		=2,

	/**
	 * \brief Error, info, and debug message are logged.
	 */
	LOGGER_LEVEL_DEBUG		=3,

	//just to ensure that always the same number of bytes is used for this enum even if it is extended
	__SIZE_SCALER_LL		=0xFF //do not use
} logger_loglevel_t;

/**
 * \brief An enumeration of flags used to define the scope of information, which should be sent to the application on a call
 * of automounter_api_get_snapshot().
 */
typedef enum __attribute__((packed)) snapshot_scope_t
{
	/**
	 * \brief Information about all devices and partitions currently managed by the automounter is sent to the application.
	 */
	SNAPSHOT_COMPLETE					=0,

	/**
	 * \brief Information about all partitions currently managed by the automounter is sent to the application.
	 */
	SNAPSHOT_PARTITIONS_ONLY			=1,

	/**
	 * \brief Information about all mounted partitions currently managed by the automounter is sent to the application.
	 */
	SNAPSHOT_MOUNTED_PARTITIONS_ONLY	=2,

	//just to ensure that always the same number of bytes is used for this enum even if it is extended
	__SIZE_SCALER_SS					=0xFF //do not use
} snapshot_scope_t;

#ifdef __cplusplus
}
#endif

#endif
